import nodemailer from 'nodemailer';

// Create transporter with SMTP settings from environment
let transporter = null;

function getTransporter() {
  if (transporter) return transporter;
  
  const smtpConfig = {
    host: process.env.SMTP_HOST || 'vorai.cloud',
    port: parseInt(process.env.SMTP_PORT) || 465,
    secure: process.env.SMTP_SECURE === 'true',
    auth: {
      user: process.env.SMTP_USER,
      pass: process.env.SMTP_PASSWORD,
    },
  };

  // Only create transporter if credentials are configured
  if (!smtpConfig.auth.user || !smtpConfig.auth.pass) {
    console.warn('SMTP credentials not configured. Email sending disabled.');
    return null;
  }

  transporter = nodemailer.createTransport(smtpConfig);
  return transporter;
}

// Professional email template for license key delivery
function getLicenseEmailTemplate(data) {
  const {
    customerName,
    customerEmail,
    companyName,
    licenseKey,
    tierName,
    paymentModel,
    expiresAt,
    maxActivations,
  } = data;

  const formattedExpiry = expiresAt 
    ? new Date(expiresAt).toLocaleDateString('en-US', { 
        year: 'numeric', 
        month: 'long', 
        day: 'numeric' 
      })
    : 'Never (Perpetual)';

  const paymentModelDisplay = paymentModel === 'perpetual' 
    ? 'Perpetual License' 
    : 'Subscription License';

  return {
    subject: `Your VorAI Accounting License Key - ${tierName}`,
    html: `
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Your License Key</title>
</head>
<body style="margin: 0; padding: 0; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background-color: #f4f7fa;">
  <table role="presentation" style="width: 100%; border-collapse: collapse;">
    <tr>
      <td align="center" style="padding: 40px 0;">
        <table role="presentation" style="width: 600px; max-width: 100%; border-collapse: collapse; background-color: #ffffff; border-radius: 12px; box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);">
          
          <!-- Header -->
          <tr>
            <td style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px 40px 30px; border-radius: 12px 12px 0 0; text-align: center;">
              <h1 style="margin: 0; color: #ffffff; font-size: 28px; font-weight: 700;">
                🎉 Your License Key is Ready!
              </h1>
              <p style="margin: 10px 0 0; color: rgba(255, 255, 255, 0.9); font-size: 16px;">
                VorAI Accounting - ${tierName}
              </p>
            </td>
          </tr>

          <!-- Body -->
          <tr>
            <td style="padding: 40px;">
              <p style="margin: 0 0 20px; color: #374151; font-size: 16px; line-height: 1.6;">
                Dear <strong>${customerName}</strong>,
              </p>
              
              <p style="margin: 0 0 30px; color: #374151; font-size: 16px; line-height: 1.6;">
                Thank you for choosing VorAI Accounting! Your license key has been generated and is ready for activation.
              </p>

              <!-- License Key Box -->
              <div style="background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%); border: 2px solid #0ea5e9; border-radius: 12px; padding: 25px; text-align: center; margin-bottom: 30px;">
                <p style="margin: 0 0 10px; color: #0369a1; font-size: 14px; font-weight: 600; text-transform: uppercase; letter-spacing: 1px;">
                  Your License Key
                </p>
                <p style="margin: 0; font-family: 'Courier New', monospace; font-size: 24px; font-weight: 700; color: #0c4a6e; letter-spacing: 3px; word-break: break-all;">
                  ${licenseKey}
                </p>
              </div>

              <!-- License Details -->
              <table role="presentation" style="width: 100%; border-collapse: collapse; margin-bottom: 30px;">
                <tr>
                  <td style="padding: 15px; background-color: #f9fafb; border-radius: 8px 8px 0 0; border-bottom: 1px solid #e5e7eb;">
                    <table role="presentation" style="width: 100%;">
                      <tr>
                        <td style="color: #6b7280; font-size: 14px;">License Type</td>
                        <td style="text-align: right; color: #111827; font-size: 14px; font-weight: 600;">${tierName}</td>
                      </tr>
                    </table>
                  </td>
                </tr>
                <tr>
                  <td style="padding: 15px; background-color: #f9fafb; border-bottom: 1px solid #e5e7eb;">
                    <table role="presentation" style="width: 100%;">
                      <tr>
                        <td style="color: #6b7280; font-size: 14px;">Payment Model</td>
                        <td style="text-align: right; color: #111827; font-size: 14px; font-weight: 600;">${paymentModelDisplay}</td>
                      </tr>
                    </table>
                  </td>
                </tr>
                <tr>
                  <td style="padding: 15px; background-color: #f9fafb; border-bottom: 1px solid #e5e7eb;">
                    <table role="presentation" style="width: 100%;">
                      <tr>
                        <td style="color: #6b7280; font-size: 14px;">Valid Until</td>
                        <td style="text-align: right; color: #111827; font-size: 14px; font-weight: 600;">${formattedExpiry}</td>
                      </tr>
                    </table>
                  </td>
                </tr>
                <tr>
                  <td style="padding: 15px; background-color: #f9fafb; border-radius: 0 0 8px 8px;">
                    <table role="presentation" style="width: 100%;">
                      <tr>
                        <td style="color: #6b7280; font-size: 14px;">Max Activations</td>
                        <td style="text-align: right; color: #111827; font-size: 14px; font-weight: 600;">${maxActivations} device(s)</td>
                      </tr>
                    </table>
                  </td>
                </tr>
              </table>

              <!-- Activation Instructions -->
              <div style="background-color: #fef3c7; border-left: 4px solid #f59e0b; padding: 20px; border-radius: 0 8px 8px 0; margin-bottom: 30px;">
                <h3 style="margin: 0 0 10px; color: #92400e; font-size: 16px; font-weight: 600;">
                  📋 How to Activate
                </h3>
                <ol style="margin: 0; padding-left: 20px; color: #78350f; font-size: 14px; line-height: 1.8;">
                  <li>Open VorAI Accounting application</li>
                  <li>Go to <strong>Settings → License</strong></li>
                  <li>Enter the license key above</li>
                  <li>Click <strong>Activate</strong></li>
                </ol>
              </div>

              <!-- Support -->
              <p style="margin: 0; color: #6b7280; font-size: 14px; line-height: 1.6;">
                Need help? Contact our support team at <a href="mailto:support@vorai.cloud" style="color: #667eea; text-decoration: none;">support@vorai.cloud</a>
              </p>
            </td>
          </tr>

          <!-- Footer -->
          <tr>
            <td style="background-color: #f9fafb; padding: 30px 40px; border-radius: 0 0 12px 12px; text-align: center; border-top: 1px solid #e5e7eb;">
              <p style="margin: 0 0 10px; color: #6b7280; font-size: 14px;">
                Thank you for your business!
              </p>
              <p style="margin: 0; color: #9ca3af; font-size: 12px;">
                © ${new Date().getFullYear()} VorAI - Dynamic Solutions. All rights reserved.
              </p>
              <p style="margin: 10px 0 0; color: #9ca3af; font-size: 12px;">
                This is an automated message. Please do not reply directly to this email.
              </p>
            </td>
          </tr>

        </table>
      </td>
    </tr>
  </table>
</body>
</html>
    `,
    text: `
Your VorAI Accounting License Key

Dear ${customerName},

Thank you for choosing VorAI Accounting! Your license key has been generated and is ready for activation.

YOUR LICENSE KEY:
${licenseKey}

LICENSE DETAILS:
- License Type: ${tierName}
- Payment Model: ${paymentModelDisplay}
- Valid Until: ${formattedExpiry}
- Max Activations: ${maxActivations} device(s)

HOW TO ACTIVATE:
1. Open VorAI Accounting application
2. Go to Settings → License
3. Enter the license key above
4. Click Activate

Need help? Contact our support team at support@vorai.cloud

Thank you for your business!

© ${new Date().getFullYear()} VorAI - Dynamic Solutions. All rights reserved.
    `,
  };
}

// Send license key email
async function sendLicenseKeyEmail(data) {
  const transport = getTransporter();
  
  if (!transport) {
    throw new Error('SMTP not configured. Please set SMTP_USER and SMTP_PASSWORD in environment variables.');
  }

  const { customerEmail, customerName } = data;
  const template = getLicenseEmailTemplate(data);

  const mailOptions = {
    from: `"${process.env.SMTP_FROM_NAME || 'VorAI Licensing'}" <${process.env.SMTP_FROM_EMAIL || process.env.SMTP_USER}>`,
    to: customerEmail,
    subject: template.subject,
    html: template.html,
    text: template.text,
  };

  try {
    const result = await transport.sendMail(mailOptions);
    console.log(`License key email sent to ${customerEmail}. MessageId: ${result.messageId}`);
    return { success: true, messageId: result.messageId };
  } catch (error) {
    console.error('Failed to send license key email:', error);
    throw new Error(`Failed to send email: ${error.message}`);
  }
}

// Test SMTP connection
async function testSmtpConnection() {
  const transport = getTransporter();
  
  if (!transport) {
    return { success: false, error: 'SMTP not configured' };
  }

  try {
    await transport.verify();
    return { success: true };
  } catch (error) {
    return { success: false, error: error.message };
  }
}

export {
  sendLicenseKeyEmail,
  testSmtpConnection,
  getLicenseEmailTemplate,
};
