import express from 'express';
import db from '../db.js';
import { 
  generateLicenseKey, 
  validateLicenseKeyFormat, 
  generateActivationToken,
  verifyActivationToken 
} from '../license-utils.js';
import { sendLicenseKeyEmail } from '../services/emailService.js';

const router = express.Router();

/**
 * Activate a license key
 * POST /api/license/activate
 * Body: { licenseKey, machineId, machineName }
 */
router.post('/activate', async (req, res) => {
  try {
    const { licenseKey, machineId, machineName } = req.body;
    const ipAddress = req.ip || req.connection.remoteAddress;
    const userAgent = req.headers['user-agent'];

    if (!licenseKey || !machineId) {
      return res.status(400).json({ 
        success: false, 
        error: 'License key and machine ID are required' 
      });
    }

    const upperKey = licenseKey.toUpperCase().trim();

    // Validate format
    if (!validateLicenseKeyFormat(upperKey)) {
      await logActivity(null, 'activation_failed', machineId, ipAddress, { 
        reason: 'invalid_format', 
        key: upperKey 
      });
      return res.status(400).json({ 
        success: false, 
        error: 'Invalid license key format' 
      });
    }

    // Find the license key
    const keyResult = await db.query(
      'SELECT * FROM license_keys WHERE license_key = $1',
      [upperKey]
    );

    if (keyResult.rows.length === 0) {
      await logActivity(null, 'activation_failed', machineId, ipAddress, { 
        reason: 'key_not_found', 
        key: upperKey 
      });
      return res.status(400).json({ 
        success: false, 
        error: 'License key not found' 
      });
    }

    const license = keyResult.rows[0];

    // Check if license is active
    if (!license.is_active) {
      await logActivity(license.id, 'activation_failed', machineId, ipAddress, { 
        reason: 'key_disabled' 
      });
      return res.status(400).json({ 
        success: false, 
        error: 'This license key has been disabled' 
      });
    }

    // Check if license has expired
    if (license.expires_at && new Date(license.expires_at) < new Date()) {
      await logActivity(license.id, 'activation_failed', machineId, ipAddress, { 
        reason: 'key_expired' 
      });
      return res.status(400).json({ 
        success: false, 
        error: 'This license key has expired' 
      });
    }

    // Check if already activated on this machine
    const existingActivation = await db.query(
      'SELECT * FROM activations WHERE license_key_id = $1 AND machine_id = $2 AND is_active = true',
      [license.id, machineId]
    );

    if (existingActivation.rows.length > 0) {
      const activation = existingActivation.rows[0];
      
      // Update last check time
      await db.query(
        'UPDATE activations SET last_check_at = CURRENT_TIMESTAMP WHERE id = $1',
        [activation.id]
      );

      const token = generateActivationToken(upperKey, machineId, activation.expires_at);

      return res.json({
        success: true,
        message: 'License already activated on this machine',
        licenseType: license.license_type,
        paymentModel: license.payment_model || 'subscription',
        expiresAt: activation.expires_at,
        token
      });
    }

    // Count current activations
    const activationCount = await db.query(
      'SELECT COUNT(*) as count FROM activations WHERE license_key_id = $1 AND is_active = true',
      [license.id]
    );

    const currentCount = parseInt(activationCount.rows[0].count);

    if (currentCount >= license.max_activations) {
      await logActivity(license.id, 'activation_failed', machineId, ipAddress, { 
        reason: 'max_activations_reached',
        current: currentCount,
        max: license.max_activations
      });
      return res.status(400).json({ 
        success: false, 
        error: `Maximum activations reached (${license.max_activations})` 
      });
    }

    // Calculate expiration for this activation
    let expiresAt = null;
    if (license.valid_days > 0) {
      expiresAt = new Date();
      expiresAt.setDate(expiresAt.getDate() + license.valid_days);
    }

    // Create activation
    await db.query(
      `INSERT INTO activations (license_key_id, machine_id, machine_name, ip_address, user_agent, expires_at, is_active)
       VALUES ($1, $2, $3, $4, $5, $6, true)`,
      [license.id, machineId, machineName || 'Unknown', ipAddress, userAgent, expiresAt]
    );

    await logActivity(license.id, 'activated', machineId, ipAddress, { 
      machine_name: machineName,
      expires_at: expiresAt
    });

    const token = generateActivationToken(upperKey, machineId, expiresAt);

    res.json({
      success: true,
      message: 'License activated successfully',
      licenseType: license.license_type,
      paymentModel: license.payment_model || 'subscription',
      expiresAt,
      token
    });

  } catch (error) {
    console.error('Activation error:', error);
    res.status(500).json({ success: false, error: 'Server error during activation' });
  }
});

/**
 * Verify/validate an activation
 * POST /api/license/verify
 * Body: { licenseKey, machineId } OR { token }
 */
router.post('/verify', async (req, res) => {
  try {
    const { licenseKey, machineId, token } = req.body;
    const ipAddress = req.ip || req.connection.remoteAddress;

    // If token provided, verify it first (for offline grace period)
    if (token) {
      const tokenResult = verifyActivationToken(token);
      if (tokenResult.valid) {
        // Token is valid, but let's also verify with database if possible
        const { key, mid } = tokenResult.payload;
        
        const result = await verifyActivationInDb(key, mid, ipAddress);
        if (result.valid) {
          return res.json(result);
        }
        
        // If DB check fails but token is valid, allow grace period
        return res.json({
          valid: true,
          gracePeriod: true,
          licenseType: 'unknown',
          message: 'Offline validation - limited functionality'
        });
      }
    }

    // Direct verification
    if (!licenseKey || !machineId) {
      return res.status(400).json({ 
        valid: false, 
        error: 'License key and machine ID required' 
      });
    }

    const result = await verifyActivationInDb(licenseKey.toUpperCase(), machineId, ipAddress);
    res.json(result);

  } catch (error) {
    console.error('Verification error:', error);
    res.status(500).json({ valid: false, error: 'Server error during verification' });
  }
});

/**
 * Deactivate a license on a machine
 * POST /api/license/deactivate
 */
router.post('/deactivate', async (req, res) => {
  try {
    const { licenseKey, machineId, token } = req.body;
    const ipAddress = req.ip || req.connection.remoteAddress;

    let key = licenseKey?.toUpperCase();
    let mid = machineId;

    // If token provided, extract info from it
    if (token && !key) {
      const tokenResult = verifyActivationToken(token);
      if (tokenResult.valid) {
        key = tokenResult.payload.key;
        mid = tokenResult.payload.mid;
      }
    }

    if (!key || !mid) {
      return res.status(400).json({ 
        success: false, 
        error: 'License key and machine ID required' 
      });
    }

    // Find license
    const keyResult = await db.query(
      'SELECT id FROM license_keys WHERE license_key = $1',
      [key]
    );

    if (keyResult.rows.length === 0) {
      return res.status(400).json({ success: false, error: 'License key not found' });
    }

    const licenseId = keyResult.rows[0].id;

    // Deactivate
    await db.query(
      'UPDATE activations SET is_active = false WHERE license_key_id = $1 AND machine_id = $2',
      [licenseId, mid]
    );

    await logActivity(licenseId, 'deactivated', mid, ipAddress, {});

    res.json({ success: true, message: 'License deactivated' });

  } catch (error) {
    console.error('Deactivation error:', error);
    res.status(500).json({ success: false, error: 'Server error during deactivation' });
  }
});

/**
 * Get license status (for checking on app startup)
 * POST /api/license/status
 */
router.post('/status', async (req, res) => {
  try {
    const { token } = req.body;

    if (!token) {
      return res.json({ licensed: false });
    }

    const tokenResult = verifyActivationToken(token);
    
    if (!tokenResult.valid) {
      return res.json({ licensed: false, error: tokenResult.error });
    }

    const { key, mid, exp } = tokenResult.payload;

    // Verify in database
    const result = await verifyActivationInDb(key, mid, req.ip);
    
    if (result.valid) {
      return res.json({
        licensed: true,
        licenseType: result.licenseType,
        paymentModel: result.paymentModel,
        expiresAt: result.expiresAt,
        daysRemaining: result.daysRemaining,
        isPerpetual: result.isPerpetual
      });
    }

    // If DB check fails but token hasn't expired, allow grace period
    if (exp && exp > Date.now()) {
      return res.json({
        licensed: true,
        gracePeriod: true,
        licenseType: 'unknown',
        expiresAt: new Date(exp).toISOString()
      });
    }

    res.json({ licensed: false });

  } catch (error) {
    console.error('Status check error:', error);
    res.status(500).json({ licensed: false, error: 'Server error' });
  }
});

// Helper function to verify activation in database
async function verifyActivationInDb(licenseKey, machineId, ipAddress) {
  const result = await db.query(
    `SELECT a.*, lk.license_type, lk.payment_model, lk.is_active as key_active, lk.expires_at as key_expires
     FROM activations a
     JOIN license_keys lk ON a.license_key_id = lk.id
     WHERE lk.license_key = $1 AND a.machine_id = $2 AND a.is_active = true`,
    [licenseKey, machineId]
  );

  if (result.rows.length === 0) {
    return { valid: false, error: 'Activation not found' };
  }

  const activation = result.rows[0];
  const paymentModel = activation.payment_model || 'subscription';

  // Check if license key is still active
  if (!activation.key_active) {
    return { valid: false, error: 'License key has been disabled' };
  }

  // Check if activation has expired (only for subscription licenses)
  if (paymentModel !== 'one-time' && activation.expires_at && new Date(activation.expires_at) < new Date()) {
    return { valid: false, error: 'License has expired' };
  }

  // Update last check time
  await db.query(
    'UPDATE activations SET last_check_at = CURRENT_TIMESTAMP WHERE id = $1',
    [activation.id]
  );

  // Calculate days remaining (null for perpetual licenses)
  let daysRemaining = null;
  if (paymentModel !== 'one-time' && activation.expires_at) {
    daysRemaining = Math.ceil((new Date(activation.expires_at) - new Date()) / (1000 * 60 * 60 * 24));
  }

  // Get features for this license tier
  const featuresResult = await db.query(`
    SELECT tf.feature_key 
    FROM tier_features tf
    JOIN license_tiers lt ON tf.tier_id = lt.id
    WHERE lt.tier_key = $1 AND tf.is_enabled = true
  `, [activation.license_type]);
  
  const features = featuresResult.rows.map(f => f.feature_key);

  // Get tier limits
  const tierResult = await db.query(`
    SELECT max_invoices, max_customers, max_expenses, max_users
    FROM license_tiers WHERE tier_key = $1
  `, [activation.license_type]);
  
  const tierLimits = tierResult.rows[0] || {};

  return {
    valid: true,
    licenseType: activation.license_type,
    paymentModel: paymentModel,
    expiresAt: paymentModel === 'one-time' ? null : activation.expires_at,
    daysRemaining,
    isPerpetual: paymentModel === 'one-time',
    features,
    limits: {
      maxInvoices: tierLimits.max_invoices,
      maxCustomers: tierLimits.max_customers,
      maxExpenses: tierLimits.max_expenses,
      maxUsers: tierLimits.max_users
    }
  };
}

// Helper to log activity
async function logActivity(licenseKeyId, action, machineId, ipAddress, details) {
  try {
    await db.query(
      `INSERT INTO activity_log (license_key_id, action, machine_id, ip_address, details)
       VALUES ($1, $2, $3, $4, $5)`,
      [licenseKeyId, action, machineId, ipAddress, JSON.stringify(details)]
    );
  } catch (error) {
    console.error('Failed to log activity:', error);
  }
}

/**
 * Get public pricing data for landing page
 * GET /api/license/pricing
 */
router.get('/pricing', async (req, res) => {
  try {
    const tiers = await db.query(`
      SELECT tier_key, name, description, icon, color, price_monthly, price_yearly, price_perpetual,
             max_invoices, max_customers, max_expenses, max_users, sort_order
      FROM license_tiers 
      WHERE is_active = true
      ORDER BY sort_order, id
    `);

    // Get features for each tier
    const pricingData = [];
    
    for (const tier of tiers.rows) {
      const features = await db.query(`
        SELECT f.feature_key, f.name, f.description 
        FROM tier_features tf
        JOIN available_features f ON tf.feature_key = f.feature_key
        WHERE tf.tier_id = (SELECT id FROM license_tiers WHERE tier_key = $1)
        AND tf.is_enabled = true
        ORDER BY f.sort_order
      `, [tier.tier_key]);

      pricingData.push({
        key: tier.tier_key,
        name: tier.name,
        description: tier.description,
        icon: tier.icon,
        color: tier.color,
        priceMonthly: tier.price_monthly,
        priceYearly: tier.price_yearly,
        pricePerpetual: tier.price_perpetual,
        maxInvoices: tier.max_invoices,
        maxCustomers: tier.max_customers,
        maxExpenses: tier.max_expenses,
        maxUsers: tier.max_users,
        features: features.rows.map(f => ({
          key: f.feature_key,
          name: f.name,
          description: f.description
        }))
      });
    }

    res.json(pricingData);
  } catch (error) {
    console.error('Get pricing error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Get tier configuration (public endpoint for client app)
 * GET /api/license/tiers
 */
router.get('/tiers', async (req, res) => {
  try {
    const tiers = await db.query(`
      SELECT tier_key, name, icon, color, max_invoices, max_customers, max_expenses, max_users
      FROM license_tiers 
      WHERE is_active = true
      ORDER BY sort_order, id
    `);

    // Get features for each tier
    const tiersConfig = {};
    
    for (const tier of tiers.rows) {
      const features = await db.query(`
        SELECT feature_key FROM tier_features 
        WHERE tier_id = (SELECT id FROM license_tiers WHERE tier_key = $1)
        AND is_enabled = true
      `, [tier.tier_key]);

      tiersConfig[tier.tier_key] = {
        name: tier.name,
        icon: tier.icon,
        color: tier.color,
        maxInvoices: tier.max_invoices,
        maxCustomers: tier.max_customers,
        maxExpenses: tier.max_expenses,
        maxUsers: tier.max_users,
        features: features.rows.map(f => f.feature_key)
      };
    }

    res.json(tiersConfig);
  } catch (error) {
    console.error('Get tiers config error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Request a free trial license (public endpoint)
 * POST /api/license/request-trial
 * Body: { email, name, companyName }
 */
router.post('/request-trial', async (req, res) => {
  try {
    const { email, name, companyName } = req.body;
    const ipAddress = req.ip || req.connection.remoteAddress;

    if (!email) {
      return res.status(400).json({ error: 'Email is required' });
    }

    // Validate email format
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
      return res.status(400).json({ error: 'Invalid email format' });
    }

    // Check if email already has a trial
    const existingCustomer = await db.query(
      'SELECT id FROM customers WHERE email = $1',
      [email.toLowerCase()]
    );

    if (existingCustomer.rows.length > 0) {
      // Check if they already have a trial key
      const existingTrial = await db.query(
        `SELECT lk.license_key FROM license_keys lk 
         WHERE lk.customer_id = $1 AND lk.license_type = 'trial'`,
        [existingCustomer.rows[0].id]
      );

      if (existingTrial.rows.length > 0) {
        return res.status(400).json({ 
          error: 'A trial license has already been issued to this email address' 
        });
      }
    }

    // Create or get customer
    let customerId;
    if (existingCustomer.rows.length > 0) {
      customerId = existingCustomer.rows[0].id;
    } else {
      const customerResult = await db.query(
        `INSERT INTO customers (company_name, contact_name, email)
         VALUES ($1, $2, $3) RETURNING id`,
        [companyName || name || 'Trial User', name || 'Trial User', email.toLowerCase()]
      );
      customerId = customerResult.rows[0].id;
    }

    // Generate trial license key (30 days)
    const licenseKey = generateLicenseKey();
    const expiresAt = new Date();
    expiresAt.setDate(expiresAt.getDate() + 30);

    await db.query(
      `INSERT INTO license_keys (license_key, customer_id, license_type, payment_model, max_activations, valid_days, expires_at, notes)
       VALUES ($1, $2, 'trial', 'subscription', 1, 30, $3, $4)`,
      [licenseKey, customerId, expiresAt, `Trial requested from website - IP: ${ipAddress}`]
    );

    // Send the trial key via email
    try {
      await sendLicenseKeyEmail({
        customerName: name || 'Valued Customer',
        customerEmail: email,
        companyName: companyName || '',
        licenseKey: licenseKey,
        tierName: 'Trial',
        paymentModel: 'trial',
        expiresAt: expiresAt,
        maxActivations: 1,
      });

      res.json({ 
        success: true, 
        message: 'Trial license key has been sent to your email!' 
      });
    } catch (emailError) {
      console.error('Failed to send trial email:', emailError);
      // Still return success but mention email issue
      res.json({ 
        success: true, 
        message: 'Trial license created! If you don\'t receive an email, please contact support.',
        licenseKey: licenseKey // Include key in response as fallback
      });
    }

  } catch (error) {
    console.error('Trial request error:', error);
    res.status(500).json({ error: 'Failed to process trial request' });
  }
});

export default router;
