import express from 'express';
import bcrypt from 'bcryptjs';
import db from '../db.js';
import { authenticateToken, generateToken } from '../auth.js';
import { generateLicenseKey } from '../license-utils.js';
import { sendLicenseKeyEmail, testSmtpConnection } from '../services/emailService.js';

const router = express.Router();

// ==================== AUTH ====================

/**
 * Admin login
 * POST /api/admin/login
 */
router.post('/login', async (req, res) => {
  try {
    const { email, password } = req.body;

    if (!email || !password) {
      return res.status(400).json({ error: 'Email and password required' });
    }

    const result = await db.query('SELECT * FROM admins WHERE email = $1', [email]);

    if (result.rows.length === 0) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    const admin = result.rows[0];
    const validPassword = await bcrypt.compare(password, admin.password);

    if (!validPassword) {
      return res.status(401).json({ error: 'Invalid credentials' });
    }

    const token = generateToken(admin);

    res.json({
      token,
      admin: {
        id: admin.id,
        email: admin.email,
        name: admin.name,
        role: admin.role
      }
    });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Get current admin
 * GET /api/admin/me
 */
router.get('/me', authenticateToken, async (req, res) => {
  try {
    const result = await db.query(
      'SELECT id, email, name, role FROM admins WHERE id = $1',
      [req.admin.id]
    );
    res.json(result.rows[0]);
  } catch (error) {
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Update admin profile (email, name)
 * PUT /api/admin/profile
 */
router.put('/profile', authenticateToken, async (req, res) => {
  try {
    const { email, name } = req.body;
    
    if (!email) {
      return res.status(400).json({ error: 'Email is required' });
    }

    // Check if email is already taken by another admin
    const existing = await db.query(
      'SELECT id FROM admins WHERE email = $1 AND id != $2',
      [email, req.admin.id]
    );
    
    if (existing.rows.length > 0) {
      return res.status(400).json({ error: 'Email is already in use' });
    }

    const result = await db.query(
      'UPDATE admins SET email = $1, name = $2 WHERE id = $3 RETURNING id, email, name, role',
      [email, name || null, req.admin.id]
    );

    res.json(result.rows[0]);
  } catch (error) {
    console.error('Update profile error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Update admin password
 * PUT /api/admin/password
 */
router.put('/password', authenticateToken, async (req, res) => {
  try {
    const { currentPassword, newPassword } = req.body;
    
    if (!currentPassword || !newPassword) {
      return res.status(400).json({ error: 'Current password and new password are required' });
    }

    if (newPassword.length < 6) {
      return res.status(400).json({ error: 'New password must be at least 6 characters' });
    }

    // Get current admin with password
    const adminResult = await db.query(
      'SELECT * FROM admins WHERE id = $1',
      [req.admin.id]
    );

    if (adminResult.rows.length === 0) {
      return res.status(404).json({ error: 'Admin not found' });
    }

    const admin = adminResult.rows[0];

    // Verify current password
    const validPassword = await bcrypt.compare(currentPassword, admin.password);
    if (!validPassword) {
      return res.status(400).json({ error: 'Current password is incorrect' });
    }

    // Hash new password and update
    const hashedPassword = await bcrypt.hash(newPassword, 10);
    await db.query(
      'UPDATE admins SET password = $1 WHERE id = $2',
      [hashedPassword, req.admin.id]
    );

    res.json({ success: true, message: 'Password updated successfully' });
  } catch (error) {
    console.error('Update password error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// ==================== CUSTOMERS ====================

/**
 * Get all customers
 */
router.get('/customers', authenticateToken, async (req, res) => {
  try {
    const result = await db.query(`
      SELECT c.*, 
        (SELECT COUNT(*) FROM license_keys lk WHERE lk.customer_id = c.id) as license_count,
        (SELECT COUNT(*) FROM license_keys lk 
         JOIN activations a ON a.license_key_id = lk.id 
         WHERE lk.customer_id = c.id AND a.is_active = true) as active_activations
      FROM customers c
      ORDER BY c.created_at DESC
    `);
    res.json(result.rows);
  } catch (error) {
    console.error('Get customers error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Create customer
 */
router.post('/customers', authenticateToken, async (req, res) => {
  try {
    const { companyName, contactName, email, phone, address, notes } = req.body;

    const result = await db.query(
      `INSERT INTO customers (company_name, contact_name, email, phone, address, notes)
       VALUES ($1, $2, $3, $4, $5, $6) RETURNING *`,
      [companyName, contactName, email, phone, address, notes]
    );

    res.status(201).json(result.rows[0]);
  } catch (error) {
    if (error.code === '23505') {
      return res.status(400).json({ error: 'Customer with this email already exists' });
    }
    console.error('Create customer error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Update customer
 */
router.put('/customers/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const { companyName, contactName, email, phone, address, notes, isActive } = req.body;

    const result = await db.query(
      `UPDATE customers SET 
        company_name = $1, contact_name = $2, email = $3, phone = $4, 
        address = $5, notes = $6, is_active = $7, updated_at = CURRENT_TIMESTAMP
       WHERE id = $8 RETURNING *`,
      [companyName, contactName, email, phone, address, notes, isActive, id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Customer not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    console.error('Update customer error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Delete customer
 */
router.delete('/customers/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    await db.query('DELETE FROM customers WHERE id = $1', [id]);
    res.json({ success: true });
  } catch (error) {
    console.error('Delete customer error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// ==================== LICENSE KEYS ====================

/**
 * Get all license keys
 */
router.get('/licenses', authenticateToken, async (req, res) => {
  try {
    const result = await db.query(`
      SELECT lk.*, 
        c.company_name, c.email as customer_email,
        (SELECT COUNT(*) FROM activations a WHERE a.license_key_id = lk.id AND a.is_active = true) as activation_count
      FROM license_keys lk
      LEFT JOIN customers c ON lk.customer_id = c.id
      ORDER BY lk.created_at DESC
    `);
    res.json(result.rows);
  } catch (error) {
    console.error('Get licenses error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Generate new license key(s)
 */
router.post('/licenses', authenticateToken, async (req, res) => {
  try {
    const { 
      customerId, 
      licenseType = 'standard', 
      paymentModel = 'subscription',
      maxActivations = 1, 
      validDays = 365, 
      notes,
      count = 1 
    } = req.body;

    if (count < 1 || count > 100) {
      return res.status(400).json({ error: 'Count must be between 1 and 100' });
    }

    // Validate payment model
    if (!['subscription', 'one-time'].includes(paymentModel)) {
      return res.status(400).json({ error: 'Invalid payment model' });
    }

    const licenses = [];

    for (let i = 0; i < count; i++) {
      const licenseKey = generateLicenseKey();
      
      // One-time (perpetual) licenses never expire
      let expiresAt = null;
      let effectiveValidDays = validDays;
      
      if (paymentModel === 'one-time') {
        // Perpetual license - never expires
        effectiveValidDays = 0;
        expiresAt = null;
      } else if (validDays > 0) {
        // Subscription license with expiry
        expiresAt = new Date();
        expiresAt.setDate(expiresAt.getDate() + validDays);
      }

      const result = await db.query(
        `INSERT INTO license_keys (license_key, customer_id, license_type, payment_model, max_activations, valid_days, expires_at, notes)
         VALUES ($1, $2, $3, $4, $5, $6, $7, $8) RETURNING *`,
        [licenseKey, customerId || null, licenseType, paymentModel, maxActivations, effectiveValidDays, expiresAt, notes]
      );

      licenses.push(result.rows[0]);
    }

    res.status(201).json(count === 1 ? licenses[0] : licenses);
  } catch (error) {
    console.error('Generate license error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Update license key
 */
router.put('/licenses/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const { customerId, licenseType, maxActivations, isActive, notes } = req.body;

    const result = await db.query(
      `UPDATE license_keys SET 
        customer_id = $1, license_type = $2, max_activations = $3, is_active = $4, notes = $5
       WHERE id = $6 RETURNING *`,
      [customerId, licenseType, maxActivations, isActive, notes, id]
    );

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'License not found' });
    }

    res.json(result.rows[0]);
  } catch (error) {
    console.error('Update license error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Delete license key
 */
router.delete('/licenses/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    await db.query('DELETE FROM license_keys WHERE id = $1', [id]);
    res.json({ success: true });
  } catch (error) {
    console.error('Delete license error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Get activations for a license
 */
router.get('/licenses/:id/activations', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const result = await db.query(
      `SELECT * FROM activations WHERE license_key_id = $1 ORDER BY activated_at DESC`,
      [id]
    );
    res.json(result.rows);
  } catch (error) {
    console.error('Get activations error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Revoke an activation
 */
router.delete('/activations/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    await db.query('UPDATE activations SET is_active = false WHERE id = $1', [id]);
    res.json({ success: true });
  } catch (error) {
    console.error('Revoke activation error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// ==================== EMAIL ====================

/**
 * Send license key to customer email
 * POST /api/admin/licenses/:id/send-email
 */
router.post('/licenses/:id/send-email', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    
    // Get license with customer info
    const licenseResult = await db.query(`
      SELECT 
        lk.*,
        c.contact_name as customer_name,
        c.email as customer_email,
        c.company_name
      FROM license_keys lk
      JOIN customers c ON lk.customer_id = c.id
      WHERE lk.id = $1
    `, [id]);
    
    if (licenseResult.rows.length === 0) {
      return res.status(404).json({ error: 'License not found' });
    }
    
    const license = licenseResult.rows[0];
    
    if (!license.customer_email) {
      return res.status(400).json({ error: 'Customer has no email address' });
    }
    
    // Send the email
    const result = await sendLicenseKeyEmail({
      customerName: license.customer_name || 'Valued Customer',
      customerEmail: license.customer_email,
      companyName: license.company_name,
      licenseKey: license.license_key,
      tierName: license.license_type || 'Standard',
      paymentModel: license.payment_model,
      expiresAt: license.expires_at,
      maxActivations: license.max_activations,
    });
    
    // Log the activity
    await db.query(`
      INSERT INTO activity_log (license_key_id, action, ip_address, details)
      VALUES ($1, 'email_sent', $2, $3)
    `, [id, req.ip, JSON.stringify({ 
      email: license.customer_email, 
      messageId: result.messageId 
    })]);
    
    res.json({ 
      success: true, 
      message: `License key sent to ${license.customer_email}`,
      messageId: result.messageId
    });
    
  } catch (error) {
    console.error('Send email error:', error);
    res.status(500).json({ error: error.message || 'Failed to send email' });
  }
});

/**
 * Test SMTP connection
 * POST /api/admin/email/test
 */
router.post('/email/test', authenticateToken, async (req, res) => {
  try {
    const result = await testSmtpConnection();
    if (result.success) {
      res.json({ success: true, message: 'SMTP connection successful' });
    } else {
      res.status(400).json({ success: false, error: result.error });
    }
  } catch (error) {
    console.error('Test SMTP error:', error);
    res.status(500).json({ error: error.message });
  }
});

// ==================== ACTIVITY LOG ====================

/**
 * Get activity log
 */
router.get('/activity', authenticateToken, async (req, res) => {
  try {
    const { limit = 100, licenseId, action } = req.query;
    
    let query = `
      SELECT al.*, lk.license_key
      FROM activity_log al
      LEFT JOIN license_keys lk ON al.license_key_id = lk.id
    `;
    const params = [];
    const conditions = [];

    if (licenseId) {
      conditions.push(`al.license_key_id = $${params.length + 1}`);
      params.push(parseInt(licenseId));
    }

    if (action) {
      conditions.push(`al.action = $${params.length + 1}`);
      params.push(action);
    }

    if (conditions.length > 0) {
      query += ' WHERE ' + conditions.join(' AND ');
    }

    query += ` ORDER BY al.created_at DESC LIMIT $${params.length + 1}`;
    params.push(parseInt(limit) || 100);

    const result = await db.query(query, params);
    res.json(result.rows);
  } catch (error) {
    console.error('Get activity error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// ==================== DASHBOARD STATS ====================

/**
 * Get dashboard statistics
 */
router.get('/stats', authenticateToken, async (req, res) => {
  try {
    const [
      totalCustomers,
      totalLicenses,
      activeLicenses,
      totalActivations,
      recentActivity
    ] = await Promise.all([
      db.query('SELECT COUNT(*) FROM customers'),
      db.query('SELECT COUNT(*) FROM license_keys'),
      db.query('SELECT COUNT(*) FROM license_keys WHERE is_active = true'),
      db.query('SELECT COUNT(*) FROM activations WHERE is_active = true'),
      db.query(`
        SELECT action, COUNT(*) as count 
        FROM activity_log 
        WHERE created_at > NOW() - INTERVAL '24 hours'
        GROUP BY action
      `)
    ]);

    // Licenses by type
    const licensesByType = await db.query(`
      SELECT license_type, COUNT(*) as count 
      FROM license_keys 
      GROUP BY license_type
    `);

    // Activations over time (last 30 days)
    const activationsOverTime = await db.query(`
      SELECT DATE(activated_at) as date, COUNT(*) as count
      FROM activations
      WHERE activated_at > NOW() - INTERVAL '30 days'
      GROUP BY DATE(activated_at)
      ORDER BY date
    `);

    res.json({
      totalCustomers: parseInt(totalCustomers.rows[0].count),
      totalLicenses: parseInt(totalLicenses.rows[0].count),
      activeLicenses: parseInt(activeLicenses.rows[0].count),
      totalActivations: parseInt(totalActivations.rows[0].count),
      recentActivity: recentActivity.rows,
      licensesByType: licensesByType.rows,
      activationsOverTime: activationsOverTime.rows
    });
  } catch (error) {
    console.error('Get stats error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// ==================== LICENSE TIERS ====================

/**
 * Get all license tiers with their features
 */
router.get('/tiers', authenticateToken, async (req, res) => {
  try {
    const tiers = await db.query(`
      SELECT * FROM license_tiers 
      ORDER BY sort_order, id
    `);

    // Get features for each tier
    const tiersWithFeatures = await Promise.all(
      tiers.rows.map(async (tier) => {
        const features = await db.query(`
          SELECT feature_key FROM tier_features 
          WHERE tier_id = $1 AND is_enabled = true
        `, [tier.id]);
        return {
          ...tier,
          features: features.rows.map(f => f.feature_key)
        };
      })
    );

    res.json(tiersWithFeatures);
  } catch (error) {
    console.error('Get tiers error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Create a new tier
 */
router.post('/tiers', authenticateToken, async (req, res) => {
  try {
    const {
      tierKey, name, description, icon, color,
      maxInvoices, maxCustomers, maxExpenses, maxUsers,
      priceMonthly, priceYearly, pricePerpetual, sortOrder, features
    } = req.body;

    if (!tierKey || !name) {
      return res.status(400).json({ error: 'Tier key and name are required' });
    }

    // Convert empty strings to null for numeric fields
    const cleanPriceMonthly = priceMonthly === '' || priceMonthly === undefined ? null : priceMonthly;
    const cleanPriceYearly = priceYearly === '' || priceYearly === undefined ? null : priceYearly;
    const cleanPricePerpetual = pricePerpetual === '' || pricePerpetual === undefined ? null : pricePerpetual;
    const cleanMaxInvoices = maxInvoices === '' || maxInvoices === undefined ? -1 : parseInt(maxInvoices);
    const cleanMaxCustomers = maxCustomers === '' || maxCustomers === undefined ? -1 : parseInt(maxCustomers);
    const cleanMaxExpenses = maxExpenses === '' || maxExpenses === undefined ? -1 : parseInt(maxExpenses);
    const cleanMaxUsers = maxUsers === '' || maxUsers === undefined ? 1 : parseInt(maxUsers);
    const cleanSortOrder = sortOrder === '' || sortOrder === undefined ? 0 : parseInt(sortOrder);

    const result = await db.query(`
      INSERT INTO license_tiers 
      (tier_key, name, description, icon, color, max_invoices, max_customers, max_expenses, max_users, price_monthly, price_yearly, price_perpetual, sort_order)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
      RETURNING *
    `, [tierKey, name, description, icon || '⭐', color || 'blue', 
        cleanMaxInvoices, cleanMaxCustomers, cleanMaxExpenses, cleanMaxUsers,
        cleanPriceMonthly, cleanPriceYearly, cleanPricePerpetual, cleanSortOrder]);

    const tier = result.rows[0];

    // Add features if provided
    if (features && features.length > 0) {
      for (const featureKey of features) {
        await db.query(`
          INSERT INTO tier_features (tier_id, feature_key, is_enabled)
          VALUES ($1, $2, true)
          ON CONFLICT (tier_id, feature_key) DO UPDATE SET is_enabled = true
        `, [tier.id, featureKey]);
      }
    }

    res.status(201).json({ ...tier, features: features || [] });
  } catch (error) {
    console.error('Create tier error:', error);
    if (error.code === '23505') {
      return res.status(400).json({ error: 'Tier key already exists' });
    }
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Update a tier
 */
router.put('/tiers/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;
    const {
      tierKey, name, description, icon, color,
      maxInvoices, maxCustomers, maxExpenses, maxUsers,
      priceMonthly, priceYearly, pricePerpetual, sortOrder, isActive, features
    } = req.body;

    // Convert empty strings to null for numeric fields
    const cleanPriceMonthly = priceMonthly === '' || priceMonthly === undefined ? null : priceMonthly;
    const cleanPriceYearly = priceYearly === '' || priceYearly === undefined ? null : priceYearly;
    const cleanPricePerpetual = pricePerpetual === '' || pricePerpetual === undefined ? null : pricePerpetual;
    const cleanMaxInvoices = maxInvoices === '' || maxInvoices === undefined ? null : parseInt(maxInvoices);
    const cleanMaxCustomers = maxCustomers === '' || maxCustomers === undefined ? null : parseInt(maxCustomers);
    const cleanMaxExpenses = maxExpenses === '' || maxExpenses === undefined ? null : parseInt(maxExpenses);
    const cleanMaxUsers = maxUsers === '' || maxUsers === undefined ? null : parseInt(maxUsers);
    const cleanSortOrder = sortOrder === '' || sortOrder === undefined ? null : parseInt(sortOrder);

    const result = await db.query(`
      UPDATE license_tiers SET
        tier_key = COALESCE($1, tier_key),
        name = COALESCE($2, name),
        description = COALESCE($3, description),
        icon = COALESCE($4, icon),
        color = COALESCE($5, color),
        max_invoices = COALESCE($6, max_invoices),
        max_customers = COALESCE($7, max_customers),
        max_expenses = COALESCE($8, max_expenses),
        max_users = COALESCE($9, max_users),
        price_monthly = COALESCE($10, price_monthly),
        price_yearly = COALESCE($11, price_yearly),
        price_perpetual = COALESCE($12, price_perpetual),
        sort_order = COALESCE($13, sort_order),
        is_active = COALESCE($14, is_active),
        updated_at = CURRENT_TIMESTAMP
      WHERE id = $15
      RETURNING *
    `, [tierKey, name, description, icon, color,
        cleanMaxInvoices, cleanMaxCustomers, cleanMaxExpenses, cleanMaxUsers,
        cleanPriceMonthly, cleanPriceYearly, cleanPricePerpetual, cleanSortOrder, isActive, id]);

    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Tier not found' });
    }

    const tier = result.rows[0];

    // Update features if provided
    if (features !== undefined) {
      // Remove all existing features
      await db.query('DELETE FROM tier_features WHERE tier_id = $1', [id]);
      
      // Add new features
      for (const featureKey of features) {
        await db.query(`
          INSERT INTO tier_features (tier_id, feature_key, is_enabled)
          VALUES ($1, $2, true)
        `, [id, featureKey]);
      }
    }

    res.json({ ...tier, features: features || [] });
  } catch (error) {
    console.error('Update tier error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Delete a tier
 */
router.delete('/tiers/:id', authenticateToken, async (req, res) => {
  try {
    const { id } = req.params;

    // Check if any licenses use this tier
    const tier = await db.query('SELECT tier_key FROM license_tiers WHERE id = $1', [id]);
    if (tier.rows.length === 0) {
      return res.status(404).json({ error: 'Tier not found' });
    }

    const licenses = await db.query(
      'SELECT COUNT(*) FROM license_keys WHERE license_type = $1',
      [tier.rows[0].tier_key]
    );

    if (parseInt(licenses.rows[0].count) > 0) {
      return res.status(400).json({ 
        error: 'Cannot delete tier with active licenses. Reassign licenses first.' 
      });
    }

    await db.query('DELETE FROM license_tiers WHERE id = $1', [id]);
    res.json({ success: true });
  } catch (error) {
    console.error('Delete tier error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// ==================== AVAILABLE FEATURES ====================

/**
 * Get all available features
 */
router.get('/features', authenticateToken, async (req, res) => {
  try {
    const result = await db.query(`
      SELECT * FROM available_features 
      ORDER BY category, sort_order, name
    `);
    res.json(result.rows);
  } catch (error) {
    console.error('Get features error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Create or update a feature
 */
router.post('/features', authenticateToken, async (req, res) => {
  try {
    const { featureKey, name, description, category, sortOrder } = req.body;

    if (!featureKey || !name) {
      return res.status(400).json({ error: 'Feature key and name are required' });
    }

    const result = await db.query(`
      INSERT INTO available_features (feature_key, name, description, category, sort_order)
      VALUES ($1, $2, $3, $4, $5)
      ON CONFLICT (feature_key) DO UPDATE SET
        name = $2, description = $3, category = $4, sort_order = $5
      RETURNING *
    `, [featureKey, name, description, category, sortOrder ?? 0]);

    res.json(result.rows[0]);
  } catch (error) {
    console.error('Create/update feature error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Delete a feature
 */
router.delete('/features/:key', authenticateToken, async (req, res) => {
  try {
    const { key } = req.params;
    await db.query('DELETE FROM tier_features WHERE feature_key = $1', [key]);
    await db.query('DELETE FROM available_features WHERE feature_key = $1', [key]);
    res.json({ success: true });
  } catch (error) {
    console.error('Delete feature error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Seed default features (one-time setup)
 */
router.post('/features/seed', authenticateToken, async (req, res) => {
  try {
    const defaultFeatures = [
      { key: 'dashboard', name: 'Dashboard', category: 'Core', order: 1 },
      { key: 'invoices', name: 'Invoices', category: 'Sales', order: 2 },
      { key: 'estimates', name: 'Estimates', category: 'Sales', order: 3 },
      { key: 'creditNotes', name: 'Credit Notes', category: 'Sales', order: 4 },
      { key: 'customers', name: 'Customers', category: 'Sales', order: 5 },
      { key: 'leads', name: 'Leads', category: 'Sales', order: 6 },
      { key: 'expenses', name: 'Expenses', category: 'Purchases', order: 7 },
      { key: 'bills', name: 'Bills', category: 'Purchases', order: 8 },
      { key: 'vendors', name: 'Vendors', category: 'Purchases', order: 9 },
      { key: 'purchaseOrders', name: 'Purchase Orders', category: 'Purchases', order: 10 },
      { key: 'inventory', name: 'Inventory', category: 'Operations', order: 11 },
      { key: 'projects', name: 'Projects', category: 'Operations', order: 12 },
      { key: 'timeTracking', name: 'Time Tracking', category: 'Operations', order: 13 },
      { key: 'employees', name: 'Employees', category: 'HR & Payroll', order: 14 },
      { key: 'payroll', name: 'Payroll', category: 'HR & Payroll', order: 15 },
      { key: 'bankReconciliation', name: 'Bank Reconciliation', category: 'Accounting', order: 16 },
      { key: 'chartOfAccounts', name: 'Chart of Accounts', category: 'Accounting', order: 17 },
      { key: 'paymentReminders', name: 'Payment Reminders', category: 'Accounting', order: 18 },
      { key: 'ubtFiling', name: 'UBT Filing', category: 'Accounting', order: 19 },
      { key: 'reports', name: 'Reports', category: 'Reports', order: 20 },
      { key: 'templates', name: 'Templates', category: 'Settings', order: 21 },
      { key: 'import', name: 'Import', category: 'Settings', order: 22 },
      { key: 'settings', name: 'Settings', category: 'Settings', order: 23 },
      { key: 'guesthouse', name: 'Guesthouse Module', category: 'Modules', order: 24 },
    ];

    for (const f of defaultFeatures) {
      await db.query(`
        INSERT INTO available_features (feature_key, name, category, sort_order)
        VALUES ($1, $2, $3, $4)
        ON CONFLICT (feature_key) DO NOTHING
      `, [f.key, f.name, f.category, f.order]);
    }

    res.json({ success: true, message: `Seeded ${defaultFeatures.length} features` });
  } catch (error) {
    console.error('Seed features error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

/**
 * Seed default tiers (one-time setup)
 */
router.post('/tiers/seed', authenticateToken, async (req, res) => {
  try {
    const defaultTiers = [
      {
        tierKey: 'none', name: 'No License', icon: '🔒', color: 'gray', sortOrder: 0,
        maxInvoices: 3, maxCustomers: 5, maxExpenses: 5,
        features: ['dashboard', 'invoices', 'customers', 'expenses', 'settings']
      },
      {
        tierKey: 'trial', name: 'Trial', icon: '⏱️', color: 'yellow', sortOrder: 1,
        maxInvoices: 10, maxCustomers: 10, maxExpenses: 20,
        features: ['dashboard', 'invoices', 'estimates', 'customers', 'expenses', 'reports', 'settings']
      },
      {
        tierKey: 'standard', name: 'Standard', icon: '⭐', color: 'blue', sortOrder: 2,
        maxInvoices: -1, maxCustomers: -1, maxExpenses: -1,
        features: ['dashboard', 'invoices', 'estimates', 'creditNotes', 'customers', 'leads', 
                   'expenses', 'bills', 'vendors', 'reports', 'templates', 'import', 'settings',
                   'chartOfAccounts', 'paymentReminders', 'inventory', 'bankReconciliation', 'purchaseOrders']
      },
      {
        tierKey: 'professional', name: 'Professional', icon: '💎', color: 'purple', sortOrder: 3,
        maxInvoices: -1, maxCustomers: -1, maxExpenses: -1,
        features: ['dashboard', 'invoices', 'estimates', 'creditNotes', 'customers', 'leads',
                   'expenses', 'bills', 'vendors', 'purchaseOrders', 'inventory', 'projects',
                   'timeTracking', 'employees', 'payroll', 'bankReconciliation', 'chartOfAccounts',
                   'paymentReminders', 'ubtFiling', 'reports', 'templates', 'import', 'settings', 'guesthouse']
      },
      {
        tierKey: 'enterprise', name: 'Enterprise', icon: '🏆', color: 'amber', sortOrder: 4,
        maxInvoices: -1, maxCustomers: -1, maxExpenses: -1,
        features: ['dashboard', 'invoices', 'estimates', 'creditNotes', 'customers', 'leads',
                   'expenses', 'bills', 'vendors', 'purchaseOrders', 'inventory', 'projects',
                   'timeTracking', 'employees', 'payroll', 'bankReconciliation', 'chartOfAccounts',
                   'paymentReminders', 'ubtFiling', 'reports', 'templates', 'import', 'settings', 'guesthouse']
      }
    ];

    for (const t of defaultTiers) {
      const result = await db.query(`
        INSERT INTO license_tiers (tier_key, name, icon, color, sort_order, max_invoices, max_customers, max_expenses)
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        ON CONFLICT (tier_key) DO UPDATE SET
          name = $2, icon = $3, color = $4, sort_order = $5,
          max_invoices = $6, max_customers = $7, max_expenses = $8
        RETURNING id
      `, [t.tierKey, t.name, t.icon, t.color, t.sortOrder, t.maxInvoices, t.maxCustomers, t.maxExpenses]);

      const tierId = result.rows[0].id;

      // Clear and re-add features
      await db.query('DELETE FROM tier_features WHERE tier_id = $1', [tierId]);
      for (const featureKey of t.features) {
        await db.query(`
          INSERT INTO tier_features (tier_id, feature_key, is_enabled)
          VALUES ($1, $2, true)
        `, [tierId, featureKey]);
      }
    }

    res.json({ success: true, message: `Seeded ${defaultTiers.length} tiers` });
  } catch (error) {
    console.error('Seed tiers error:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

export default router;
