import crypto from 'crypto';

const LICENSE_SECRET = process.env.LICENSE_SECRET || 'VORAI-LICENSE-SECRET-2024';

/**
 * Generate a valid license key
 * Format: XXXX-XXXX-XXXX-XXXX-XXXX
 * Last segment is a checksum for basic offline validation
 */
export function generateLicenseKey() {
  const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
  const segments = [];
  
  // Generate 4 random segments
  for (let i = 0; i < 4; i++) {
    let segment = '';
    for (let j = 0; j < 4; j++) {
      segment += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    segments.push(segment);
  }
  
  // Generate checksum segment
  const baseKey = segments.join('-');
  const hash = crypto.createHmac('sha256', LICENSE_SECRET)
    .update(baseKey)
    .digest('hex')
    .substring(0, 4)
    .toUpperCase();
  
  segments.push(hash);
  return segments.join('-');
}

/**
 * Validate a license key format and checksum
 */
export function validateLicenseKeyFormat(key) {
  if (!key) return false;
  
  // Check format: XXXX-XXXX-XXXX-XXXX-XXXX
  const pattern = /^[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/;
  if (!pattern.test(key.toUpperCase())) {
    return false;
  }
  
  const upperKey = key.toUpperCase();
  const segments = upperKey.split('-');
  const providedChecksum = segments[4];
  const baseKey = segments.slice(0, 4).join('-');
  
  // Verify checksum
  const expectedChecksum = crypto.createHmac('sha256', LICENSE_SECRET)
    .update(baseKey)
    .digest('hex')
    .substring(0, 4)
    .toUpperCase();
  
  return providedChecksum === expectedChecksum;
}

/**
 * Generate an activation token (signed JWT-like token for the client to store)
 */
export function generateActivationToken(licenseKey, machineId, expiresAt) {
  const payload = {
    key: licenseKey,
    mid: machineId,
    exp: expiresAt ? new Date(expiresAt).getTime() : null,
    iat: Date.now()
  };
  
  const data = Buffer.from(JSON.stringify(payload)).toString('base64');
  const signature = crypto.createHmac('sha256', LICENSE_SECRET)
    .update(data)
    .digest('hex');
  
  return `${data}.${signature}`;
}

/**
 * Verify an activation token
 */
export function verifyActivationToken(token) {
  try {
    const [data, signature] = token.split('.');
    
    const expectedSignature = crypto.createHmac('sha256', LICENSE_SECRET)
      .update(data)
      .digest('hex');
    
    if (signature !== expectedSignature) {
      return { valid: false, error: 'Invalid token signature' };
    }
    
    const payload = JSON.parse(Buffer.from(data, 'base64').toString());
    
    // Check expiration
    if (payload.exp && payload.exp < Date.now()) {
      return { valid: false, error: 'Token expired', payload };
    }
    
    return { valid: true, payload };
  } catch (error) {
    return { valid: false, error: 'Invalid token format' };
  }
}
