import pg from 'pg';
import dotenv from 'dotenv';

dotenv.config();

const { Pool } = pg;

const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
  ssl: process.env.NODE_ENV === 'production' ? { rejectUnauthorized: false } : false
});

// Initialize database tables
export async function initDatabase() {
  const client = await pool.connect();
  
  try {
    // Create tables
    await client.query(`
      -- Admin users table
      CREATE TABLE IF NOT EXISTS admins (
        id SERIAL PRIMARY KEY,
        email VARCHAR(255) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        name VARCHAR(255),
        role VARCHAR(50) DEFAULT 'admin',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );

      -- Customers/Companies table
      CREATE TABLE IF NOT EXISTS customers (
        id SERIAL PRIMARY KEY,
        company_name VARCHAR(255) NOT NULL,
        contact_name VARCHAR(255),
        email VARCHAR(255) UNIQUE NOT NULL,
        phone VARCHAR(50),
        address TEXT,
        notes TEXT,
        is_active BOOLEAN DEFAULT true,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );

      -- License keys table
      CREATE TABLE IF NOT EXISTS license_keys (
        id SERIAL PRIMARY KEY,
        license_key VARCHAR(30) UNIQUE NOT NULL,
        customer_id INTEGER REFERENCES customers(id) ON DELETE SET NULL,
        license_type VARCHAR(50) DEFAULT 'standard',
        payment_model VARCHAR(20) DEFAULT 'subscription',
        max_activations INTEGER DEFAULT 1,
        valid_days INTEGER DEFAULT 365,
        is_active BOOLEAN DEFAULT true,
        notes TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        expires_at TIMESTAMP
      );

      -- Add payment_model column if it doesn't exist (for existing databases)
      DO $$ 
      BEGIN 
        IF NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'license_keys' AND column_name = 'payment_model') THEN
          ALTER TABLE license_keys ADD COLUMN payment_model VARCHAR(20) DEFAULT 'subscription';
        END IF;
      END $$;

      -- Activations table (tracks where licenses are activated)
      CREATE TABLE IF NOT EXISTS activations (
        id SERIAL PRIMARY KEY,
        license_key_id INTEGER REFERENCES license_keys(id) ON DELETE CASCADE,
        machine_id VARCHAR(64) NOT NULL,
        machine_name VARCHAR(255),
        ip_address VARCHAR(45),
        user_agent TEXT,
        activated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        last_check_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        expires_at TIMESTAMP,
        is_active BOOLEAN DEFAULT true,
        UNIQUE(license_key_id, machine_id)
      );

      -- License activity log
      CREATE TABLE IF NOT EXISTS activity_log (
        id SERIAL PRIMARY KEY,
        license_key_id INTEGER REFERENCES license_keys(id) ON DELETE SET NULL,
        action VARCHAR(50) NOT NULL,
        machine_id VARCHAR(64),
        ip_address VARCHAR(45),
        details JSONB,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );

      -- Create indexes
      CREATE INDEX IF NOT EXISTS idx_license_keys_key ON license_keys(license_key);
      CREATE INDEX IF NOT EXISTS idx_activations_license ON activations(license_key_id);
      CREATE INDEX IF NOT EXISTS idx_activations_machine ON activations(machine_id);
      CREATE INDEX IF NOT EXISTS idx_activity_log_license ON activity_log(license_key_id);

      -- License tiers configuration table
      CREATE TABLE IF NOT EXISTS license_tiers (
        id SERIAL PRIMARY KEY,
        tier_key VARCHAR(50) UNIQUE NOT NULL,
        name VARCHAR(100) NOT NULL,
        description TEXT,
        icon VARCHAR(10) DEFAULT '⭐',
        color VARCHAR(50) DEFAULT 'blue',
        max_invoices INTEGER DEFAULT -1,
        max_customers INTEGER DEFAULT -1,
        max_expenses INTEGER DEFAULT -1,
        max_users INTEGER DEFAULT 1,
        price_monthly DECIMAL(10,2),
        price_yearly DECIMAL(10,2),
        price_perpetual DECIMAL(10,2),
        sort_order INTEGER DEFAULT 0,
        is_active BOOLEAN DEFAULT true,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );

      -- Tier features (many-to-many relationship)
      CREATE TABLE IF NOT EXISTS tier_features (
        id SERIAL PRIMARY KEY,
        tier_id INTEGER REFERENCES license_tiers(id) ON DELETE CASCADE,
        feature_key VARCHAR(100) NOT NULL,
        is_enabled BOOLEAN DEFAULT true,
        UNIQUE(tier_id, feature_key)
      );

      -- Available features registry
      CREATE TABLE IF NOT EXISTS available_features (
        id SERIAL PRIMARY KEY,
        feature_key VARCHAR(100) UNIQUE NOT NULL,
        name VARCHAR(255) NOT NULL,
        description TEXT,
        category VARCHAR(100),
        sort_order INTEGER DEFAULT 0,
        is_active BOOLEAN DEFAULT true
      );

      -- Create indexes for tiers
      CREATE INDEX IF NOT EXISTS idx_tier_features_tier ON tier_features(tier_id);
      CREATE INDEX IF NOT EXISTS idx_license_tiers_key ON license_tiers(tier_key);
    `);

    // Run migrations for existing databases
    await client.query(`
      -- Add price_perpetual column if it doesn't exist
      DO $$ 
      BEGIN 
        IF NOT EXISTS (SELECT 1 FROM information_schema.columns 
                       WHERE table_name = 'license_tiers' AND column_name = 'price_perpetual') THEN
          ALTER TABLE license_tiers ADD COLUMN price_perpetual DECIMAL(10,2);
        END IF;
      END $$;
    `);

    console.log('✅ Database tables initialized');
  } finally {
    client.release();
  }
}

export async function query(text, params) {
  const start = Date.now();
  const res = await pool.query(text, params);
  const duration = Date.now() - start;
  if (process.env.NODE_ENV === 'development') {
    console.log('Executed query', { text: text.substring(0, 50), duration, rows: res.rowCount });
  }
  return res;
}

export async function getClient() {
  return await pool.connect();
}

export default { query, getClient, initDatabase };
